#
# -*- Perl -*-
# $Id: pltests.pl.in,v 1.4.2.8 2006/01/21 20:03:16 opengl2772 Exp $
# Copyright (C) 2004-2006 Tadamasa Teranishi
#               2004-2006 Namazu Project All rights reserved.
#     This is free software with ABSOLUTELY NO WARRANTY.
#
#  This program is free software; you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation; either versions 2, or (at your option)
#  any later version.
#
#  This program is distributed in the hope that it will be useful
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program; if not, write to the Free Software
#  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
#  02111-1307, USA
#
#  This file must be encoded in EUC-JP encoding
#
#

package pltests;
use strict;
use English;
require Cwd;
use IO::File;
use File::Basename;
use File::Copy;
use NKF;

sub cdie (@);

if ($English::OSNAME eq "MSWin32" || $English::OSNAME eq "os2") {
    cdie 'Error: "pkgdatadir": Undefined environment variable.' unless(defined $ENV{'pkgdatadir'});
} else {
    eval '$ENV{pkgdatadir} = "C:/namazu/share/namazu" unless(defined $ENV{pkgdatadir})';
}

sub selectdata() {
    my $cwd = Cwd::cwd();
    my $base = "$cwd/../tests/data";
    my @lang = ("LC_ALL", "LC_CTYPE", "LANG");
  
    for my $lang (@lang) {
        $lang = $ENV{$lang};
        if (defined $lang) {
            if ($lang eq "C") {
                return "$base/en";
            } elsif ($lang =~ /^(..)/) {
                my $dir = "$base/$1";
                if (-d "$dir") {
                    return $dir;
                }
            }
        }
    }

    return "$base/en";
}

sub get_lang()
{
    my @lang = ("LC_ALL", "LC_CTYPE", "LANG");

    for my $lang (@lang) {
        $lang = $ENV{$lang};
        return $lang if (defined $lang);
    }

    return 'C';
}

sub get_lang_messages()
{
    my @lang = ("LANGUAGE", "LC_ALL", "LC_MESSAGES", "LANG");

    for my $lang (@lang) {
        $lang = $ENV{$lang};
        return $lang if (defined $lang);
    }

    return 'C';
}

sub normalize_nl($)
{
    my ($conts) = @_;

    $$conts =~ s/\x0d\x0a/\x0a/g;  # Windows
    $$conts =~ s/\x0d/\x0a/g;      # Mac
    $$conts =~ s/\x0a/\n/g;
    $$conts;
}

sub putline($$;$) {
    my ($file, $msg, $args) = @_;
    my @args = @{$args} if (defined $args);

    my $fh = new IO::File ">>$file";
    
    if (!(defined $fh)) {
        warn "Can not open file. : $file";
        return '';
    }
    $fh->autoflush(1);

    if (defined $args) {
        for my $arg (@args) {
            if ($English::OSNAME eq "MSWin32" || $English::OSNAME eq "os2") {
                if ($arg !~ /^".*"$/) {
                    $arg =~ s!/!\\!g;
                }
            }
            if ($arg =~ /[\*\? ]/) {
                if ($arg !~ /^".*"$/) {
                    $arg = "\"$arg\"";
                }
            }
        }
        if ($English::OSNAME eq "MSWin32" || $English::OSNAME eq "os2") {
            if ($args[0] =~ m/(?:mknmz|gcnmz)\.bat$/i) {
                @args = ("perl", @args);
            } elsif ($args[0] =~ m/\.bat$/i) {
                my $comspec = "cmd";
                $comspec = $ENV{'COMSPEC'} if (defined $ENV{'COMSPEC'});
                if ($comspec =~ m/command\.com$/i) {
                    $comspec = pltests::checkcmd('win95cmd.exe');
                    unless (defined $comspec) {
                        cdie 'win95cmd.exe not found.';
                    }
                    $ENV{'COMSPEC'} = $comspec;
                }
                @args = ($comspec, "/d", "/x", "/c", @args);
            }
        }
        print $fh "$msg";
        for my $arg (@args) {
            print $fh "$arg ";
        }
        print $fh "\n";
    } else {
        print $fh "$msg\n";
    }
    $fh->close();

    return 1;
}

sub cat($) {
    my ($file) = @_;

    my $data = '';

    if (!open(IN, "$file")) {
        warn "Can not open file. : $file";
        return undef;
    }
    while(my $line = <IN>) {
        $data .= $line;
    }
    close(IN);

    return $data;
}

sub localpath($) {
    my ($path) = @_;

    $path =~ s!/!\\!g if (($English::OSNAME eq "MSWin32") || ($English::OSNAME eq "os2"));

    return $path;
}

sub binpath($) {
    my ($cmd) = @_;
    my $path = undef;

    if ($English::OSNAME eq "MSWin32") {
        if ($cmd =~ /namazu.cgi/) {
            $path = "../libexec/$cmd.exe";  # Windows 9x,Me need ".exe"
        } elsif ($cmd =~ /.namazurc/) {
            $path = "../libexec/$cmd";
        } else {
            if (-x "../bin/$cmd.exe") {
                $path = "../bin/$cmd.exe";
            } elsif (-x "../bin/$cmd.bat") {
                $path = "../bin/$cmd.bat";
            } else {
                $path = "../bin/$cmd";
            }
        }
    } else {
        if ($cmd =~ /(?:mknmz|gcnmz)/) {
            $path = "../scripts/$cmd";
        } else {
            if (-x "../src/.libs/$cmd") {
                $path = "../src/.libs/$cmd";
            } elsif (-x "../src/_libs/$cmd") {
                $path = "../src/_libs/$cmd";
            } else {
                $path = "../src/$cmd";
            }
        }
    }

    return localpath($path);
}

sub duprcs($) {
    my ($rc) = @_;

    if ($English::OSNAME eq "MSWin32") {
        ;
    } else {
        if (-d "../src/.libs") {
            copy($rc, "../src/.libs");
        }
        if (-d "../src/_libs") {
            copy($rc, "../src/_libs");
        }
    }
}

sub checkcmd($) {
    my $cmd = shift;
    my $pd = ':';
    $pd = ';' if (($English::OSNAME eq "MSWin32") || ($English::OSNAME eq "os2"));

    for my $dir (split(/$pd/, $ENV{'PATH'})) {
        next if ($dir eq '');
        win32_yen_to_slash(\$dir);
        return "$dir/$cmd" if (-x "$dir/$cmd" && ! -d "$dir/$cmd");
        return "$dir/$cmd" if (-x "$dir/$cmd.com" &&
                (($English::OSNAME eq "MSWin32") || ($English::OSNAME eq "os2")));
        return "$dir/$cmd" if (-x "$dir/$cmd.exe" &&
                (($English::OSNAME eq "MSWin32") || ($English::OSNAME eq "os2")));
        return "$dir/$cmd" if (-x "$dir/$cmd.bat" &&
                               ($English::OSNAME eq "MSWin32"));
        return "$dir/$cmd" if (-x "$dir/$cmd.cmd" &&
                               ($English::OSNAME eq "os2"));
    }
    return undef;
}

# cdie ... clean files before die
sub cdie (@) {
    my (@msgs)  = @_;

    print STDERR @msgs;
    print STDERR "\n" unless $msgs[$#msgs] =~ /\n$/;
    exit 2;
}

sub systemcmd(@) {
    my (@cmd) = @_;

    system { $cmd[0] } @cmd;
}

sub logsyscmd($$;$)
{
    my $status = undef;
    my ($file, $args, $env) = @_;

    $status = syscmd(
        command => $args,
        option => { 
            "stdout" => "$file",
            "stderr" => "$file",
            "mode_stdout" => "at",  # append, text
            "mode_stderr" => "at",  # append, text
        },
        env => $env,
    );

    return $status;
}

sub ezsyscmd($;$)
{
    my $status = undef;
    my ($args, $env) = @_;
    my $conts_out = "";
    my $conts_err = "";
    
    my %args = (
        command => $args,
        option => { 
            "stdout" => \$conts_out,
            "stderr" => \$conts_err,
            "mode_stdout" => "wt",  # new, text
            "mode_stderr" => "wt",  # new, text
        },
        env => $env,
    );
    $status = syscmd(%args);

    return ($status, $conts_out, $conts_err);
}

sub syscmd(%)
{
    my $status = undef;
    my %arg = @_;
    my @args = @{$arg{command}} if (defined $arg{command});
    my %option = %{$arg{option}} if (defined $arg{option});
    my %env = %{$arg{env}} if (defined $arg{env});
 
    # default option
    $option{stdout} = '/dev/null' unless(defined $option{stdout});
    $option{stderr} = '/dev/null' unless(defined $option{stderr});
    $option{mode_stdout} = 'wt' unless(defined $option{mode_stdout});
    $option{mode_stderr} = 'wt' unless(defined $option{mode_stderr});
    $option{maxsize} = -1 unless(defined $option{maxsize});

    my $handle_out = undef;
    my $handle_err = undef;
    if (ref $option{stdout}) {
        if ($option{stdout} =~ /^(IO::File|FileHandle)/) {
            $handle_out = $option{stdout};
        }
    }
    if (ref $option{stderr}) {
        if ($option{stderr} =~ /^(IO::File|FileHandle)/) {
            $handle_err = $option{stderr};
        }
    }

    my $same = 0;
    if ($option{stdout} eq $option{stderr}) {
        $same = 1;
    }
 
    my $mode_stdout;
    my $mode_stderr;
    if ($option{mode_stdout} =~ /^w/i) {
        $mode_stdout = '>';
    } elsif ($option{mode_stdout} =~ /^a/i) {
        $mode_stdout = '>>';
    } else {
        warn "unknown mode. : " . quotemeta($option{mode_stdout});
        $mode_stdout = '>>';
    }
    if ($option{mode_stderr} =~ /^w/i) {
        $mode_stderr = '>';
    } elsif ($option{mode_stderr} =~ /^a/i) {
        $mode_stderr = '>>';
    } else {
        warn "unknown mode. : " . quotemeta($option{mode_stderr});
        $mode_stderr = '>>';
    }

    my $text_stdout = undef;
    my $text_stderr = undef;
    if ($option{mode_stdout} =~ /^.t/i) {
        $text_stdout = 1;
    }
    if ($option{mode_stderr} =~ /^.t/i) {
        $text_stderr = 1;
    }

    my $arg;
    if ($English::OSNAME eq "MSWin32" || $English::OSNAME eq "os2") {
        for $arg (@args) {
            if ($arg !~ /^".*"$/) {
                $arg =~ s!/!\\!g;
                if ($arg =~ /[\*\?]/) {
                    $arg = "\"$arg\"";
                }
            }
        }
        if ($args[0] =~ m/(?:mknmz|gcnmz)\.bat$/i) {
            @args = ("perl", @args);
        } elsif ($args[0] =~ m/\.bat$/i) {
            my $comspec = "cmd";
            $comspec = $ENV{'COMSPEC'} if (defined $ENV{'COMSPEC'});
            if ($comspec =~ m/command\.com$/i) {
                $comspec = pltests::checkcmd('win95cmd.exe');
                unless (defined $comspec) {
                    cdie 'win95cmd.exe not found.';
                }
                $ENV{'COMSPEC'} = $comspec;
            }
            @args = ($comspec, "/d", "/x", "/c", @args);
        }
    } else {
        for $arg (@args) {
            $arg =~ s/^"(.*)"$/$1/;
        }
    }

    my $fh_out = undef;
    my $fh_err = undef;

    if (defined $handle_out) {
        $fh_out = $handle_out;
    } else {
        $fh_out= IO::File->new_tmpfile();
    }
    if ($same) {
        $fh_err = $fh_out;
    } else {
        if (defined $handle_err) {
            $fh_err = $handle_err;
        } else {
            $fh_err = IO::File->new_tmpfile();
        }
    }

    {
        my $saveout = new IO::File (">&" . STDOUT->fileno()) or cdie "Can't dup STDOUT: $!";
        my $saveerr = new IO::File (">&" . STDERR->fileno()) or cdie "Can't dup STDERR: $!";
        STDOUT->fdopen($fh_out->fileno(), 'w') or cdie "Can't open fh_out: $!";
        STDERR->fdopen($fh_err->fileno(), 'w') or cdie "Can't open fh_out: $!";

        # backup $ENV{}
        my %backup;
        my ($key, $value);
        while(($key, $value) = each %env) {
            $backup{$key} = $ENV{$key};
            if (defined $value) {
                $ENV{$key} = $value;
            } else {
                delete $ENV{$key};
            }
        }

        # Use an indirect object: see Perl Cookbook Recipe 16.2 in detail.
        $status = system { $args[0] } @args;

        # restore $ENV{}
        while(($key, $value) = each %env) {
            if (defined $backup{$key}) {
                $ENV{$key} = $backup{$key};
            } else {
                delete $ENV{$key};
            }
        }

        STDOUT->fdopen($saveout->fileno(), 'w') or cdie "Can't restore saveout: $!";
        STDERR->fdopen($saveerr->fileno(), 'w') or cdie "Can't restore saveerr: $!";
    }

    # Note that the file position of filehandles must be rewinded.
    $fh_out->seek(0, SEEK_SET) or cdie "seek: $!";
    $fh_err->seek(0, SEEK_SET) or cdie "seek: $!";

    if (!defined $handle_out) {
        if (ref($option{stdout}) ne 'SCALAR') {
            if ($option{stdout} eq '/dev/null') {
                $fh_out->close();
            } else {
                my $conts_out = "";
                my $size = -s $fh_out;
                read $fh_out, $conts_out, $size;
                $fh_out->close();
                normalize_nl(\$conts_out) if (defined $text_stdout);

                my $file = $option{stdout};
                if ($English::OSNAME eq "MSWin32" || $English::OSNAME eq "os2") {
                    $file =~ s!/!\\!g;
                }
                if (!open(OUT, "$mode_stdout$file")) {
                    warn "Can not open file. : $file";
                    return (1);
                }
                print OUT $conts_out;
                close(OUT);
            }
        } else {
            my $conts_out = $option{stdout};
            my $size = -s $fh_out;
            read $fh_out, $$conts_out, $size;
            $fh_out->close();
            normalize_nl($conts_out) if (defined $text_stdout);
        }
    }

    if (!(defined $handle_err || $same)) {
        if (ref($option{stderr}) ne 'SCALAR') {
            if ($option{stderr} eq '/dev/null') {
                $fh_err->close();
            } else {
                my $conts_err = "";
                my $size = -s $fh_err;
                read $fh_err, $conts_err, $size;
                $fh_err->close();
                normalize_nl(\$conts_err) if (defined $text_stderr);
    
                my $file = $option{stderr};
                if ($English::OSNAME eq "MSWin32" || $English::OSNAME eq "os2") {
                    $file =~ s!/!\\!g;
                }
                if (!open(OUT, "$mode_stderr$file")) {
                    warn "Can not open file. : $file";
                    return (1);
                }
                print OUT $conts_err;
                close(OUT);
            }
        } else {
            my $conts_err = $option{stderr};
            my $size = -s $fh_err;
            read $fh_err, $$conts_err, $size;
            $fh_err->close();
            normalize_nl($conts_err) if (defined $text_stderr);
        }
    }
 
    return ($status);
}

# gtnmz - program to count a total number of indexed files.
# Copyright (C) 1999 Satoru Takabayashi  All rights reserved.
#     This is free software with ABSOLUTELY NO WARRANTY.
sub gtnmz ($) {
    my ($file) = @_;

    if (!open(IN, "$file")) {
        warn "Can not open file : $file";
        return undef;
    }
    my @all   = <IN>;
    close(IN);
    @all      = grep {! /^\#\#/} @all; 
    @all      = grep {! /^$/} @all;

    my @added = grep {! /^\#/} @all;
    my @deleted  = grep {  /^\#/} @all;

    my $total = @added - @deleted;

    return $total;
}

# convert \ to / with consideration for Shift_JIS Kanji code
sub win32_yen_to_slash ($) {
    my ($filenameref) = @_;
    if (($English::OSNAME eq "MSWin32") || ($English::OSNAME eq "os2")) {
        $$filenameref =~
                s!([\x81-\x9f\xe0-\xef][\x40-\x7e\x80-\xfc]|[\x01-\x7f])!
                $1 eq "\\" ? "/" : $1!gex;
    }
}

# for Windows
#     EUC-JP -> Shift_JIS
sub codeconv_to_external($) {
    my ($contref) = @_;

    my $lang = get_lang();

    if ($lang =~ /^ja/) {
        my $nkf_opt = "-emXZ1";

        if ($lang =~ m/sjis/i) {
            $nkf_opt = "-EsmXZ1";
        }

        $$contref = NKF::nkf($nkf_opt, $$contref);
    }
}

# for Windows
#     Shift_JIS -> EUC-JP
sub codeconv_to_internal($) {
    my ($contref) = @_;

    my $lang = get_lang();

    if ($lang =~ /^ja/) {
        my $nkf_opt = "-emXZ1";

        if ($lang =~ m/sjis/i) {
            $nkf_opt = "-SemXZ1";
        }

        $$contref = NKF::nkf($nkf_opt, $$contref);
    }
}

1;
